import os
import time
import threading
from datetime import datetime
from flask import Flask, render_template, request, redirect, url_for, flash, session, send_from_directory, jsonify
from flask_sqlalchemy import SQLAlchemy
from werkzeug.security import generate_password_hash, check_password_hash
from werkzeug.utils import secure_filename
from dotenv import load_dotenv

load_dotenv()

ALLOWED_EXT = set(['pdf', 'jpg', 'jpeg'])
UPLOAD_FOLDER = os.getenv('UPLOAD_FOLDER', 'uploads')
if not os.path.isdir(UPLOAD_FOLDER):
    os.makedirs(UPLOAD_FOLDER, exist_ok=True)

app = Flask(__name__)
app.secret_key = os.getenv('SECRET_KEY', 'dev-secret-change-this')
app.config['SQLALCHEMY_DATABASE_URI'] = os.getenv('DATABASE_URL', 'mysql+pymysql://dbuser:dbpassword@localhost/document_verification')
app.config['SQLALCHEMY_TRACK_MODIFICATIONS'] = False
app.config['UPLOAD_FOLDER'] = UPLOAD_FOLDER
app.config['MAX_CONTENT_LENGTH'] = int(os.getenv('MAX_CONTENT_LENGTH', 50*1024*1024))

db = SQLAlchemy(app)

class User(db.Model):
    __tablename__ = 'users'
    id = db.Column(db.Integer, primary_key=True)
    email = db.Column(db.String(200), unique=True, nullable=False)
    password_hash = db.Column(db.String(300), nullable=False)
    created_at = db.Column(db.DateTime, default=datetime.utcnow)

    def set_password(self, raw):
        self.password_hash = generate_password_hash(raw)

    def check_password(self, raw):
        return check_password_hash(self.password_hash, raw)

class Document(db.Model):
    __tablename__ = 'documents'
    id = db.Column(db.Integer, primary_key=True)
    filename = db.Column(db.String(300), nullable=False)
    original_name = db.Column(db.String(300))
    user_id = db.Column(db.Integer, db.ForeignKey('users.id'), nullable=False)
    status = db.Column(db.String(50), default='uploaded')  # uploaded, submitted, verifying, completed
    progress = db.Column(db.Integer, default=0)
    uploaded_at = db.Column(db.DateTime, default=datetime.utcnow)
    completed_at = db.Column(db.DateTime, nullable=True)

def allowed_file(filename):
    if '.' not in filename:
        return False
    ext = filename.rsplit('.', 1)[1].lower()
    return ext in ALLOWED_EXT

def user_logged_in():
    return 'user_id' in session

def start_processing_thread(doc_id):
    def worker(did):
        doc = Document.query.get(did)
        if not doc:
            return
        doc.status = 'submitted'
        doc.progress = 10
        db.session.commit()
        for pct in range(10, 101, 10):
            time.sleep(0.4)
            doc.progress = pct
            db.session.commit()
        doc.status = 'verifying'
        doc.progress = 0
        db.session.commit()
        for pct in range(0, 101, 20):
            time.sleep(0.6)
            doc.progress = pct
            db.session.commit()
        doc.status = 'completed'
        doc.progress = 100
        doc.completed_at = datetime.utcnow()
        db.session.commit()
    t = threading.Thread(target=worker, args=(doc_id,), daemon=True)
    t.start()

@app.route('/')
def index():
    return render_template('index.html', user_email=session.get('user_email'))

@app.route('/register', methods=['POST'])
def register():
    email = request.form.get('email', '').strip().lower()
    password = request.form.get('password', '')
    if not email or not password:
        flash('Please provide email and password', 'danger')
        return redirect(url_for('index'))
    if User.query.filter_by(email=email).first():
        flash('Email already registered', 'danger')
        return redirect(url_for('index'))
    u = User(email=email)
    u.set_password(password)
    db.session.add(u)
    db.session.commit()
    flash('Registration successful. Please log in.', 'success')
    return redirect(url_for('index'))

@app.route('/login', methods=['POST'])
def login():
    email = request.form.get('username', '').strip().lower()
    password = request.form.get('password', '')
    user = User.query.filter_by(email=email).first()
    if user and user.check_password(password):
        session['user_id'] = user.id
        session['user_email'] = user.email
        flash('Login successful', 'success')
        return redirect(url_for('dashboard'))
    flash('Incorrect username or password.', 'danger')
    return redirect(url_for('index'))

@app.route('/logout')
def logout():
    session.clear()
    flash('Logged out', 'info')
    return redirect(url_for('index'))

@app.route('/dashboard')
def dashboard():
    if not user_logged_in():
        flash('Please log in', 'warning')
        return redirect(url_for('index'))
    docs = Document.query.filter_by(user_id=session['user_id']).order_by(Document.uploaded_at.desc()).all()
    return render_template('dashboard.html', docs=docs, user_email=session.get('user_email'))

@app.route('/upload', methods=['POST'])
def upload():
    if not user_logged_in():
        return jsonify({'ok': False, 'msg': 'Not authenticated'}), 403
    if 'file' not in request.files:
        return jsonify({'ok': False, 'msg': 'No file provided'}), 400
    f = request.files['file']
    if f.filename == '':
        return jsonify({'ok': False, 'msg': 'Empty filename'}), 400
    if not allowed_file(f.filename):
        return jsonify({'ok': False, 'msg': 'This is invalid file format. You can upload JPG and PDF file only'}), 400
    filename = secure_filename(f.filename)
    unique_name = str(int(time.time())) + '_' + filename
    save_path = os.path.join(app.config['UPLOAD_FOLDER'], unique_name)
    f.save(save_path)
    doc = Document(filename=unique_name, original_name=filename, user_id=session['user_id'], status='uploaded', progress=0)
    db.session.add(doc)
    db.session.commit()
    return jsonify({'ok': True, 'doc_id': doc.id, 'msg': 'File uploaded'})


@app.route('/submit/<int:doc_id>', methods=['POST'])
def submit_doc(doc_id):
    if not user_logged_in():
        return jsonify({'ok': False, 'msg': 'Not authenticated'}), 403
    doc = Document.query.get_or_404(doc_id)
    if doc.user_id != session['user_id']:
        return jsonify({'ok': False, 'msg': 'Not allowed'}), 403
    start_processing_thread(doc.id)
    return jsonify({'ok': True, 'msg': 'Submission started'})

@app.route('/status/<int:doc_id>')
def status(doc_id):
    doc = Document.query.get_or_404(doc_id)
    return jsonify({'ok': True, 'status': doc.status, 'progress': doc.progress, 'original_name': doc.original_name, 'completed_at': doc.completed_at.isoformat() if doc.completed_at else None})

@app.route('/download/<int:doc_id>')
def download(doc_id):
    doc = Document.query.get_or_404(doc_id)
    if doc.status != 'completed':
        flash('Document not ready for download', 'warning')
        return redirect(url_for('dashboard'))
    return send_from_directory(app.config['UPLOAD_FOLDER'], doc.filename, as_attachment=True, attachment_filename=doc.original_name)

@app.route('/api/current_user')
def current_user_api():
    if not user_logged_in():
        return jsonify({'logged_in': False})
    return jsonify({'logged_in': True, 'email': session.get('user_email')})

if __name__ == '__main__':
    app.run(host='0.0.0.0', debug=True)
